<?php

namespace App\Http\Controllers\Dashboard\User;

use App\Enum\ShouldTransferFail;
use App\Enum\TransferStatus;
use App\Helper\UserHelper;
use App\Http\Controllers\Controller;
use App\Mail\UserTransfer;
use App\Models\Notification;
use App\Models\Transaction;
use App\Models\Transfer;
use App\Models\TransferCode;
use App\Models\User;
use Barryvdh\DomPDF\Facade\Pdf;
use Carbon\Carbon;
use Illuminate\Support\Facades\Mail;

class TransferController extends Controller
{
    public function index()
    {
        if (UserHelper::isUserEmailVerified()) {
            return redirect()->route('user.verification')->with('error', 'Unable to verify the authenticity of this account, Please enter verification code sent to your email at the time of registration');
        }

        $user = User::findOrFail(auth()->user()->id);

        $transfers = Transfer::where('user_id', $user->id)->latest()->get();

        $data = [
            'title' => 'User transfers',
            'user'  => $user,
            'transfers' => $transfers
        ];

        return view('dashboard.user.transfer.index', $data);
    }
    public function transfer()
    {
        if (UserHelper::isUserEmailVerified()) {
            return redirect()->route('user.verification')->with('error', 'Unable to verify the authenticity of this account, Please enter verification code sent to your email at the time of registration');
        }

        $user = User::findOrFail(auth()->user()->id);
        $data = [
            'title' => 'Transfer funds',
            'user'  => $user
        ];

        return view('dashboard.user.transfer.transfer', $data);
    }
    public function preview($referenceId)
    {
        if (UserHelper::isUserEmailVerified()) {
            return redirect()->route('user.verification')->with('error', 'Unable to verify the authenticity of this account, Please enter verification code sent to your email at the time of registration');
        }

        $user = User::findOrFail(auth()->user()->id);

        $transfer = Transfer::where('reference_id', $referenceId)->first();

        $transferNeedVerificationCode   = TransferCode::where('transfer_reference_id', $referenceId)->where('user_id', $user->id)->first();

        $data = [
            'title'     => 'User transfer preview',
            'user'      => $user,
            'transfer'  => $transfer,
            'transferNeedVerificationCode' => $transferNeedVerificationCode
        ];

        return view('dashboard.user.transfer.preview', $data);
    }
    public function show($referenceId)
    {
        if (UserHelper::isUserEmailVerified()) {
            return redirect()->route('user.verification')->with('error', 'Unable to verify the authenticity of this account, Please enter verification code sent to your email at the time of registration');
        }

        $user = User::findOrFail(auth()->user()->id);

        $transfer = Transfer::where('reference_id', $referenceId)->first();

        $data = [
            'title'     => 'User transfer details',
            'user'      => $user,
            'transfer'  => $transfer
        ];

        return view('dashboard.user.transfer.show', $data);
    }
    public function confirm($referenceId)
    {
        $user = User::findOrFail(auth()->user()->id);

        $transfer = Transfer::where('reference_id', $referenceId)->first();

        $description = '';
        if ($transfer->type == "Electronic Transfer") {
            $description = 'Electronic TF: ' . $transfer->withdrawal_method . '/';
        } else {
            $description = 'Direct TF: ' . $transfer->bank_name . '/' . $transfer->account_name;
        }

        if ($user->should_transfer_fail == ShouldTransferFail::No->value) {
            $transactionData = [
                'user_id'          => $user->id,
                'transaction_type' => 'DEBIT',
                'description'      => $description,
                'amount'           => $transfer->amount,
                'transaction_date' => date('Y-m-d'),
                'transaction_time' => date('H:i:s'),
                'reference_id'     => $transfer->reference_id,
                'currency'         => currency('name'),
                'status'           => 'SUCCESSFUL'
            ];

            $user->balance = $user->balance - $transfer->amount;
            $user->save();

            $balance = $user->balance;

            $transfer->status = TransferStatus::Approved->value;
            $transfer->save();

            $transactionData['current_balance'] = $balance;
            Transaction::create($transactionData);

            $notificationMessage = '' . config('app.name') . ' Acct holder:' . $user->name . ' ' . $transactionData['transaction_type'] . ': ' . formatAmount($transactionData['amount'], 'code', $user->id) . ' Desc:' . $transactionData['description'] . ' DT:' . $transactionData['transaction_date'] . ' Available Bal:' . formatAmount($transactionData['current_balance'], 'code', $user->id) . '' . ' Status: Successful';

            $notificationData = [
                'notification_type' => $transactionData['transaction_type'],
                'notification'      => $notificationMessage,
                'user_id'           => $user->id,
            ];

            Notification::create($notificationData);

            $transaction = Transaction::where('reference_id', $transfer->reference_id)->first();

            try {
                Mail::to($user->email)->send(new UserTransfer($user, $transfer, $transaction, config('app.name') . ' ' . 'Withdrawal Notification' . Carbon::now()));
                // session()->flash('email_success', 'Email sent successfully');
            } catch (\Exception $e) {
                // session()->flash('email_error', 'An error occurred while trying to send email');
            }

            return redirect()->route('user.transfer.show', $referenceId)->with('success', 'Transfer completed successfully');
        } else {
            $transactionData = [
                'user_id'          => $user->id,
                'transaction_type' => 'DEBIT',
                'description'      => $description,
                'amount'           => $transfer->amount,
                'transaction_date' => date('Y-m-d'),
                'transaction_time' => date('H:i:s'),
                'reference_id'     => $transfer->reference_id,
                'currency'         => currency('name'),
                'status'           => 'FAILED'
            ];

            $balance = $user->balance;

            $transfer->status = TransferStatus::Failed->value;
            $transfer->save();

            $transactionData['current_balance'] = $balance;
            Transaction::create($transactionData);

            $notificationMessage = '' . config('app.name') . ' Acct holder:' . $user->name . ' ' . $transactionData['transaction_type'] . ': ' . formatAmount($transactionData['amount'], 'code', $user->id) . ' Desc:' . $transactionData['description'] . ' DT:' . $transactionData['transaction_date'] . ' Available Bal:' . formatAmount($transactionData['current_balance'], 'code', $user->id) . '' . ' Status: Failed';

            $notificationData = [
                'notification_type' => $transactionData['transaction_type'],
                'notification'      => $notificationMessage,
                'user_id'           => $user->id,
            ];

            Notification::create($notificationData);

            return redirect()->route('user.transfer.show', $referenceId)->with('error', 'Sorry we were unable to complete this transaction at the moment please try again later');
        }
    }
    public function print($referenceId)
    {
        $user = User::findOrFail(auth()->user()->id);

        $transfer = Transfer::where('user_id', $user->id)->where('reference_id', $referenceId)->first();

        $transaction = Transaction::where('reference_id', $transfer->reference_id)->where('user_id', $user->id)->first();

        $data = [
            'user'      => $user,
            'transfer'  => $transfer,
            'transaction' => $transaction
        ];

        $name = config('app.name') . '-' . 'Transfer Receipt For' . '-' . $user->name . '-' . Carbon::now();

        $pdf = Pdf::loadView('pdf.transfer', $data);

        return $pdf->download($name);
    }
    public function process($referenceId, $orderNo)
    {
        if (UserHelper::isUserEmailVerified()) {
            return redirect()->route('user.verification')->with('error', 'Unable to verify the authenticity of this account, Please enter verification code sent to your email at the time of registration');
        }

        $user = User::findOrFail(auth()->user()->id);

        $transfer = Transfer::where('reference_id', $referenceId)->first();

        $transferCode = new TransferCode();

        $data = [
            'title'         => 'User transfer process',
            'user'          => $user,
            'transfer'      => $transfer,
            'transferCodes' => $transferCode->getTransferVerificationData($transfer->reference_id),
            'referenceId'   => $referenceId,
            'orderNo'       => $orderNo
        ];

        return view('dashboard.user.transfer.process', $data);
    }
}
