<?php

namespace App\Http\Controllers\Dashboard\Admin;

use App\Http\Controllers\Controller;
use App\Mail\UserTransaction;
use App\Models\Notification;
use App\Models\Transaction;
use App\Models\User;
use Barryvdh\DomPDF\Facade\Pdf;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;

class UserTransactionController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(string $id)
    {
        $user = User::findOrFail($id);
        $transactions = Transaction::where('user_id', $user->id)->orderBy('id', 'desc')->get();

        $data = [
            'title'         => 'User Transactions',
            'user'          => $user,
            'transactions'  => $transactions,
        ];

        return view("dashboard.admin.user.transaction.index", $data);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request, $id)
    {
        $request->validate([
            'amount'       => 'required|numeric',
            'type'         => 'required',
            'date'         => 'required',
            'notification' => 'required',
            'description'  => 'required',
        ]);

        $user = User::findOrFail($id);

        $time               = date('H:i:s');
        $reference_id       = rand(55555555, 999999999);
        $currency           = currency('name', $user->id);
        $balance            = $user->balance;
        $amount             = $request->amount;


        $transactionData = [
            'user_id'          => $user->id,
            'transaction_type' => $request->type,
            'description'      => $request->description,
            'amount'           => $amount,
            'transaction_date' => $request->date,
            'transaction_time' => $time,
            'reference_id'     => $reference_id,
            'currency'         => $currency,
            'status'           => 'SUCCESSFUL'
        ];

        $emailData = [
            "accountName"       => $user->name,
            "description"       => $transactionData['description'],
            "referenceId"       => $reference_id,
            "amount"            => $amount,
            "date"              => date('dS M, Y', strtotime($transactionData['transaction_date'])),
            "currency"          => currency('code', $user->id),
            "type"              => $request->type,
            "status"            => $transactionData['status'],
        ];

        switch ($request->type) {
            case 'CREDIT':
                $user->balance = $balance + $amount;
                $user->save();

                $balance += $amount;

                $emailData["availableBalance"] = $balance;

                switch ($request->notification) {
                    case 'EMAIL':
                        try {
                            Mail::to($user->email)->send(new UserTransaction($emailData, $user, config('app.name') . ' ' . 'Deposit Notification' . Carbon::now()));
                            session()->flash('email_success', 'Email sent successfully');
                        } catch (\Exception $e) {
                            session()->flash('email_error', 'An error occurred while attempting to send the email. Please try again later.');
                        }
                        break;
                }
                break;
            case 'DEBIT':
                if ($amount > $balance) {
                    session()->flash('error', 'Transaction failed insufficient balance');
                    return redirect()->back();
                }

                $user->balance = $balance - $amount;
                $user->save();

                $balance -= $amount;

                $emailData["availableBalance"] = $balance;

                switch ($request->notification) {
                    case 'EMAIL':
                        try {
                            Mail::to($user->email)->send(new UserTransaction($emailData, $user, config('app.name') . ' ' . 'Withdrawal Notification' . Carbon::now()));
                            session()->flash('email_success', 'Email sent successfully');
                        } catch (\Exception $e) {
                            session()->flash('email_error', 'An error occurred while attempting to send the email. Please try again later.');
                        }
                        break;
                }
                break;
        }

        $transactionData['current_balance'] = $balance;
        Transaction::create($transactionData);

        $notificationMessage = '' . config('app.name') . ' Acct holder:' . $user->name . ' ' . $request->type . ': ' . formatAmount($emailData['amount'], 'code', $user->id) . ' Desc:' . $request->description . ' DT:' . $emailData['date'] . ' Available Bal:' . formatAmount($balance, 'code', $user->id) . '' . ' Status: Successful';

        $notificationData = [
            'notification_type' => $request->type,
            'notification'      => $notificationMessage,
            'user_id'           => $user->id,
        ];

        Notification::create($notificationData);

        return redirect()->back()->with('success', 'Transaction successful');
    }

    public function print(string $id, string $transactionId)
    {
        $user = User::findOrFail($id);
        $transaction = Transaction::where('user_id', $user->id)->where('id', $transactionId)->first();

        $data = [
            'user'         => $user,
            'transaction'  => $transaction
        ];

        $name = config('app.name') . '-' . 'Transaction Receipt For' . '-' . $user->name . '-' . Carbon::now();

        $pdf = Pdf::loadView('pdf.transaction', $data);

        return $pdf->download($name);
    }
    /**
     * Remove the specified resource from storage.
     */
    public function delete(string $id)
    {
        $transaction = Transaction::findOrFail($id);

        $transaction->delete();

        return redirect()->back()->with('success', 'Transaction deleted successfully');
    }
}
