<?php

namespace App\Http\Controllers\Auth;

use App\Helper\UserHelper;
use App\Http\Controllers\Controller;
use App\Mail\UserEmailVerification;
use App\Mail\Welcome;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Auth\Events\Registered;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules;
use Illuminate\View\View;
use Illuminate\Support\Facades\Mail;


class RegisteredUserController extends Controller
{
    /**
     * Display the registration view.
     */
    public function create(): View
    {
        $data = ['title' => 'Register account'];
        return view('auth.register', $data);
    }

    /**
     * Handle an incoming registration request.
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function store(Request $request): RedirectResponse
    {
        $request->validate([
            'name'          => ['required', 'string', 'max:255'],
            'email'         => ['required', 'string', 'lowercase', 'email', 'max:255', 'unique:' . User::class],
            'dial_code'     => ['required', 'string'],
            'phone'         => ['required', 'numeric', 'unique:users,phone'],
            'nationality'   => ['required', 'string'],
            'currency'      => ['required', 'string'],
            'account'       => ['required', 'string'],
            'password'      => ['required', 'confirmed', Rules\Password::defaults()],
        ]);

        $user = User::create([
            'name'              => $request->name,
            'dial_code'         => $request->dial_code,
            'phone'             => $request->phone,
            'email'             => $request->email,
            'nationality'       => $request->nationality,
            'currency'          => $request->currency,
            'account'           => $request->account,
            'account_number'    => UserHelper::generateTradingAccountNumber(11),
            'password'          => Hash::make($request->password),
        ]);

        event(new Registered($user));

        Auth::login($user);

        $content = [
            'code' => UserHelper::generateCode(),
        ];

        $user->code = $content['code'];
        $user->save();

        Mail::to($request->email)->send(new UserEmailVerification($user, $content, 'Email Verification Code Notification' . ' ' . Carbon::now()));

        Mail::to($request->email)->send(new Welcome('Welcome to ' . config('app.name') . '!', $user));

        return redirect()->route('user.verification')->with('success', 'Please enter the confirmation code sent via email, to complete your registration process.');
    }
}
