<?php

namespace App\Http\Controllers\Auth;

use Carbon\Carbon;
use App\Models\User;
use App\Mail\Welcome;
use App\Helper\UserHelper;
use Illuminate\Http\Request;
use Illuminate\Validation\Rules;
use App\Mail\UserEmailVerification;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Http\RedirectResponse;
use Illuminate\Auth\Events\Registered;




class ReferralRegistrationController extends Controller
{
    /**
     * Display the registration view.
     */
    public function create($referralCode)
    {
        $user = User::where('referral_code', $referralCode)->firstOrFail();

        if (!$user) {
            return redirect()->back()->with('error', 'Invalid referral code');
        }

        $data = [
            'title' => 'Register account',
            'referralCode' => $user->referral_code
        ];

        return view('auth.referral.register', $data);
    }
    /**
     * Handle an incoming registration request.
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function store(Request $request, $referralCode): RedirectResponse
    {
        $request->validate([
            'name'          => ['required', 'string', 'max:255'],
            'email'         => ['required', 'string', 'lowercase', 'email', 'max:255', 'unique:' . User::class],
            'nationality'   => ['required', 'string'],
            'currency'      => ['required', 'string'],
            'account'       => ['required', 'string'],
            'password'      => ['required', 'confirmed', Rules\Password::defaults()],
        ]);

        $user = User::create([
            'name'              => $request->name,
            'email'             => $request->email,
            'nationality'       => $request->nationality,
            'currency'          => $request->currency,
            'account'           => $request->account,
            'account_number'    => UserHelper::generateTradingAccountNumber(11),
            'referral'          => $referralCode,
            'password'          => Hash::make($request->password),
        ]);

        event(new Registered($user));

        Auth::login($user);

        $content = [
            'code' => UserHelper::generateCode(),
        ];

        $user->code = $content['code'];
        $user->save();

        $referralUser = User::where('referral_code', $referralCode)->first();
        $referralUser->earning += 20;
        $referralUser->save();

        Mail::to($request->email)->send(new UserEmailVerification($user, $content, 'Email Verification Code Notification' . ' ' . Carbon::now()));

        Mail::to($request->email)->send(new Welcome('Welcome to ' . config('app.name') . '!', $user));

        return redirect()->route('user.verification')->with('success', 'Please enter the confirmation code sent via email, to complete your registration process.');
    }
}
